/**
 * 
 */
package gov.va.med.mhv.usermgmt.service;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.tigris.atlas.service.ServiceResponse;

import gov.va.med.mhv.usermgmt.enumeration.BloodTypeEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.ContactMethodEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.CountryEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.GenderEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.MaritalStatusEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.PasswordHintQuestionEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.StateEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.TitleEnumeration;
import gov.va.med.mhv.usermgmt.service.delegate.PatientServiceDelegate;
import gov.va.med.mhv.usermgmt.service.delegate.ServiceDelegateFactory;
import gov.va.med.mhv.usermgmt.transfer.Patient;
import gov.va.med.mhv.usermgmt.transfer.PatientPK;
import gov.va.med.mhv.usermgmt.transfer.TransferObjectFactory;
import gov.va.med.mhv.usermgmt.transfer.UserProfile;
import gov.va.med.mhv.usermgmt.transfer.UserProfilePK;
import gov.va.med.mhv.usermgmt.util.RegistryDescriptionBuilder;

/**
 * @author Rob Proper (Aquilent Inc.)
 *
 */
public abstract class UserProfileTestCase extends BaseTestCase {
    
    private static final Log LOG = LogFactory.getLog(UserProfileTestCase.class);

    private String stagedPatientName = null;
    
    private Map<Long, UserProfile> userProfiles = 
        new HashMap<Long, UserProfile>();

    protected UserProfileTestCase() {  
        super();
    }
    protected UserProfileTestCase(String name) {  
        super(name);
    }

    protected void setUp() throws Exception {
        String key = getStagedPatientNameProperty();
        if (!StringUtils.isBlank(key)) {
            setStagedPatientName(System.getProperty(key));
        } else {
            getLog().debug("No staged patient key provided");
        }
    }

    protected void tearDown() throws Exception {
        for (UserProfile userProfile: userProfiles.values()) {
            deleteUserProfile(userProfile);
        }
    }

    protected void addOriginalUserProfile(UserProfilePK key) {
        assertNotNull("key", key);
        if (userProfiles.get(key) == null) { 
            UserProfileServiceResponse response = ServiceDelegateFactory.
                createEntityMaintenanceServiceDelegate().findByPrimaryKey(key);
            assertNoErrors(response);
            assertNotNull(response.getUserProfile());
            userProfiles.put(key.getId(), response.getUserProfile());
        } else {
            getLog().warn("Attempting to add duplicate original UserProfile" 
                + key);
            
        }
    }

    protected UserProfile getUserProfile(String userName) {
        UserProfileServiceResponse response = ServiceDelegateFactory.
            createUserProfileServiceDelegate().getProfileForUser(userName);
        assertNoErrors(response);
        return response.getUserProfile();
    }

    protected UserProfile createUserProfile(String userName, String firstName, 
        String lastName, Date birthDate, String ssn, boolean isPatient) 
    {
        UserProfile userProfile = TransferObjectFactory.createUserProfile();
        userProfile.setUserName(userName);
        userProfile.setFirstName(firstName);
        userProfile.setLastName(lastName);
        userProfile.setMiddleName("");
        userProfile.setBirthDate(birthDate);
        userProfile.setTitle(TitleEnumeration.getEnum(TitleEnumeration.MR));
        userProfile.setBloodType(BloodTypeEnumeration.getEnum(
            BloodTypeEnumeration.O_NEGATIVE));
        userProfile.setIsOrganDonor(Boolean.TRUE);
        userProfile.setGender(GenderEnumeration.getEnum(GenderEnumeration.
            MALE));
        userProfile.setMaritalStatus(MaritalStatusEnumeration.getEnum(
            MaritalStatusEnumeration.DIVORCED));
        userProfile.setIsVeteran(Boolean.TRUE);
        userProfile.setIsPatient(false);
        userProfile.setAddressStreet1("1 Some Street");
        userProfile.setAddressCity("Nowhere Town");
        userProfile.setAddressState(StateEnumeration.getEnum(StateEnumeration.
            VA));
        userProfile.setAddressPostalCode("12345");
        userProfile.setAddressCountry(CountryEnumeration.getEnum(
            CountryEnumeration.UNITED_STATES));
        userProfile.setContactInfoContactMethod(ContactMethodEnumeration.
            getEnum(ContactMethodEnumeration.HOMEPHONE));
        userProfile.setContactInfoHomePhone("555-555-5555");
        userProfile.setContactInfoEmail(firstName + "-" + lastName + 
            "-domain.ext");
        userProfile.setPasswordHintQuestion1(PasswordHintQuestionEnumeration.
            getEnum(PasswordHintQuestionEnumeration.FAVORITEFOOD));
        userProfile.setPasswordHintAnswer1("Pizza");
        userProfile.setPasswordHintQuestion2(PasswordHintQuestionEnumeration.
            getEnum(PasswordHintQuestionEnumeration.PETNAME));
        userProfile.setPasswordHintAnswer2("Spot");
        userProfile.setAcceptDisclaimer(Boolean.TRUE);
        userProfile.setAcceptPrivacy(Boolean.TRUE);
        userProfile.setAcceptTerms(Boolean.TRUE);
        userProfile.setSsn(ssn);
        userProfile.setIsPatient(isPatient);
        return userProfile;
    }

    protected void deleteUserProfile(UserProfile userProfile) {
        if (userProfile == null) {
            return;
        }
        UserProfileServiceResponse response = ServiceDelegateFactory.
            createEntityMaintenanceServiceDelegate().delete(userProfile.
             getUserProfilePK());
        if (hasErrors(response)) {
            logMessages(response, userProfile.getClass(), getLog());
        }
    }

    protected UserProfile saveUserProfile(UserProfile userProfile, 
        String messageOnError) 
    {
        assertNotNull(userProfile);
        UserProfileServiceResponse response = ServiceDelegateFactory.
            createEntityMaintenanceServiceDelegate().save(userProfile);
        if (messageOnError == null) {
            assertNoErrors(response);
        } else if (hasErrors(response)) {
            getLog().warn(messageOnError);
        }
        return response.getUserProfile();
    }

    protected Patient getPatientForUser(UserProfile userProfile) {
        assertNotNull(userProfile);
        PatientServiceResponse response = ServiceDelegateFactory.
            createPatientServiceDelegate().getPatientForUser(userProfile);
        assertNoErrors(response);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Found " + RegistryDescriptionBuilder.describe(response.
                getPatient()));
        }
        return response.getPatient();
    }

    protected Patient getPatientById(Long id) {
        assertNotNull("id", id);
        PatientPK key = TransferObjectFactory.createPatientPK();
        key.setId(id);
        PatientServiceResponse response = ServiceDelegateFactory.
            createEntityMaintenanceServiceDelegate().findByPrimaryKey(key);
        assertNoErrors(response);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Found " + RegistryDescriptionBuilder.describe(response.
                getPatient()));
        }
        return response.getPatient();
    }

    protected Patient createPatient(UserProfile userProfile, String icn) {
        assertNotNull(userProfile);
        assertNotNull(icn);
        assertTrue(userProfile.getIsPatient());
        Patient patient = TransferObjectFactory.createPatient();
        patient.setUserProfile(userProfile);
        patient.setIcn(icn);
        return patient; 
    }

    protected Patient savePatient(Patient patient, String messageOnError) {
        assertNotNull(patient);
        PatientServiceResponse response = ServiceDelegateFactory.
            createEntityMaintenanceServiceDelegate().save(patient);
        if (messageOnError == null) {
            assertNoErrors(response);
        } else if (hasErrors(response)) {
            getLog().warn(messageOnError);
        }
        return response.getPatient();
    }

    protected final Log getLog() {
        Log log = getLocalLog();
        return (log != null) ? log : LOG;
    }
    
    protected Log getLocalLog() {
        return LOG;
    }
    
    protected PatientServiceDelegate getPatientService() {
        return ServiceDelegateFactory.createPatientServiceDelegate();
    }
    
    protected String getStagedPatientNameProperty() {
        return null;
    }

    private void setStagedPatientName(String stagedUserName) {
        this.stagedPatientName = stagedUserName; 
    }
    protected String getStagedPatientName() {
        return stagedPatientName;
    }
    
    protected boolean hasStagedPatientName(String testName) {
        assertFalse(StringUtils.isBlank(testName));
        boolean hasTestUser = !StringUtils.isBlank(stagedPatientName);
        if (!hasTestUser) {
            getLog().warn("Skipping " + testName 
                + ", because no test user name specified in property '" 
                + getStagedPatientNameProperty() + "'.");
        }
        return hasTestUser;
    }

    protected Patient getStagedPatient() {
        UserProfile userProfile = getUserProfile(getStagedPatientName());
        assertNotNull("Staged Patient user profile '" + getStagedPatientName() 
            + "' not found", userProfile);
        Patient patient = getPatientForUser(userProfile);
        assertNotNull("Staged patient '" + getStagedPatientName() 
            + "' not found", patient);
        return patient;
    }
    
    protected void assertUnrecoverableExceptionOccurred(
        ServiceResponse response) 
   {
        assertNotNull(response);
        assertHasError(response, "unrecoverable.exception.occurred");

    }
}
